/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor
{
    using System;
    using System.Runtime.Serialization;
    using org.owasp.appsensor.errors;
    using Owasp.Esapi.Errors;

    /**
     * This class is the container object that holds the exception
     * that was thrown when a suspected attack occurs.  
     * Other member variables are contained, including the 
     * appsensor specific eventCode, the user who was performing the 
     * action that threw the exception, the timestamp of when the event 
     * occurred, and the location, in this case the request URI.  
     * <p>
     * These objects are managed in the IntrusionStore and used to 
     * to make decisions when thresholds are considered for specific
     * security events to determine whether or not some type of 
     * response is required. 
     * 
     * 9/2/2010 - jtmelton - Updated setters to be public - addressing
     * object reconstruction from DB - http://code.google.com/p/appsensor/issues/detail?id=1
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     */
    public class AppSensorIntrusion // TODO ISerializable
    {

        /**
         * Serial Version ID for serialization
         */
        private const long serialVersionUID = -7389012149221351603L;

        /** The exception that was thrown when an attack was suspected */
        private Exception securityException;

        /** The appsensor specific eventCode for the detection point (RE1, SE3, ACE5, etc.) */
        private String eventCode;

        /** The user associated with the exception, in this case the current user */
        private ASUser user;

        /** The time the intrusion occurred */
        private TimeSpan timeStamp;

        /** The application specific location of the occurrence, which can be used 
         * later to block requests to a given function.  In this implementation, 
         * the current request URI is used.
         */
        private String location;

        private const String DEFAULT_REQUEST_URI = "/unknown";

        /**
         * Private constructor simply for reconstruction
         */
        private AppSensorIntrusion() { }

        /**
         * Constructor that accepts the thrown exception, 
         * and fills in the remainder of the information 
         * for the intrusion. (defaults location to request URI)
         * @param e exception thrown when attack is suspected
         */
        public AppSensorIntrusion(Exception e)
        {
            construct(e);
        }

        /**
         * Constructor that accepts the thrown exception and a location, 
         * and fills in the remainder of the information 
         * for the intrusion.
         * @param e exception thrown when attack is suspected
         * @param location location to use for disabling access
         */
        public AppSensorIntrusion(Exception e, String location)
        {
            construct(e);
            Location = location;
        }

        /**
         * Helper method called from constructors to setup initial
         * data.
         * @param e exception thrown when attack is suspected
         */
        private void construct(Exception e)
        {
            setSecurityException(e);

            //setting the eventCode
            if (e is AppSensorException)
            {
                //set the event code if it's appsensor related
                AppSensorException asie = (AppSensorException)e;
                EventCode = asie.EventCode;
            }
            else if (e is AppSensorSystemException)
            {
                //set the event code if it's appsensor related
                AppSensorSystemException asie = (AppSensorSystemException)e;
                EventCode = asie.EventCode;
            }
            else
            {
                //use the class name if it's not appsensor related
                String eventCode = e.GetType().Name;
                EventCode = eventCode;
            }


            if (e is AppSensorSystemException)
            {
                User = new AppSensorSystemUser();
            }
            else
            {
                //set the user to the currently logged in user
                User = APPSENSOR.AsUtilities.CurrentUser;
            }

            //set timestamp to now
            TimeStamp = DateUtils.CurrentTime;

            //set location to request URI for this implementation
            String location = null;
            try
            {
                location = APPSENSOR.AsUtilities.GetCurrentRequest().Url.ToString();
            }
            catch (Exception)
            {
            }

            //have to grab this from the thread local since the Esapi request gets dropped sometimes.
            if (location == null)
            {
                try
                {
                    location = AppSensorServiceController.getCurrentRequestURI();
                }
                catch (Exception npe)
                {
                }
            }

            if (location == null)
            {
                location = DEFAULT_REQUEST_URI;
            }
            Location = location;
        }

        /**
         * Getter for security exception
         * @return security exception
         */
        public Exception getSecurityException()
        {
            return securityException;
        }

        /**
         * Retrieve log message from the exception
         * @return log message from the exception
         */
        public String getSecurityExceptionLogMessage()
        {
            String logMessage = "";
            if (getSecurityException() is AppSensorException)
            {
                AppSensorException e = (AppSensorException)getSecurityException();
                logMessage = e.LogMessage;
            }
            else if (getSecurityException() is AppSensorSystemException)
            {
                AppSensorSystemException e = (AppSensorSystemException)getSecurityException();
                logMessage = e.LogMessage;
            }
            else if (getSecurityException() is EnterpriseSecurityException)
            {
                EnterpriseSecurityException e = (EnterpriseSecurityException)getSecurityException();
                logMessage = e.LogMessage;
            }
            else
            {
                logMessage = getSecurityException().Message;
            }
            return logMessage;
        }

        /**
         * This method is intended to be used to reconstruct an AppSensorIntrusion object.  
         * The use case is something like the intrusions are stored in a DB or flat file and 
         * you need to reconstruct the object in order to use them in AppSensor.
         * @param securityException security exception
         * @param eventCode event code
         * @param user user
         * @param timeStamp time stamp
         * @param location location - request URI
         * @return AppSensorIntrusion object that Contains the passed in parameters as data members
         */
        public static AppSensorIntrusion reconstruct(Exception securityException,
                String eventCode, ASUser user, TimeSpan timeStamp, String location)
        {
            AppSensorIntrusion asi = new AppSensorIntrusion();
            asi.setSecurityException(securityException);
            asi.EventCode = eventCode;
            asi.User = user;
            asi.TimeStamp = timeStamp;
            asi.Location = location;
            return asi;
        }

        /**
         * Setter for security exception
         * @param securityException security exception
         */
        public void setSecurityException(Exception securityException)
        {
            this.securityException = securityException;
        }

        /**
         * Getter for event code
         * @return event code
         */
        public String EventCode
        {
            get
            {
                return eventCode;
            }
            set
            {
                eventCode = value;
            }
        }

        /**
         * Getter for user
         * @return user
         */
        public ASUser User
        {
            get
            {
                return user;
            }
            set
            {
                user = value;
            }
        }


        /**
         * Getter for timestamp
         * @return timestamp
         */
        public TimeSpan TimeStamp
        {
            get
            {
                return timeStamp;
            }
            set
            {
                timeStamp = value;
            }
        }

        /**
         * Getter for location - request URI
         * @return location - request URI
         */
        public String Location
        {
            get
            {
                return location;
            }
            set
            {
                location = value;
            }
        }
    }
}